<?php

// SPDX-FileCopyrightText: 2026 Ovation S.r.l. <dev@novamira.ai>
// SPDX-License-Identifier: AGPL-3.0-or-later

declare(strict_types=1);

function novamira_handle_sandbox_actions()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    $action = $_GET['action'] ?? null;
    $file_param = $_GET['file'] ?? null;

    if (!is_string($action) || !is_string($file_param)) {
        return;
    }

    $file = basename($file_param);
    if (!check_admin_referer('novamira_manage_file_' . $file)) {
        return;
    }

    $path = novamira_get_sandbox_dir(true) . $file;
    if (!file_exists($path)) {
        return;
    }

    $result = match ($action) {
        'delete' => unlink($path),
        'disable' => str_ends_with($file, '.php') && rename($path, $path . '.disabled'),
        'enable' => str_ends_with($file, '.disabled') && rename($path, substr($path, offset: 0, length: -9)),
        'exit_safe_mode' => $file === '.crashed' && unlink($path),
        default => false,
    };

    if ($result) {
        wp_safe_redirect(admin_url('admin.php?page=novamira-sandbox&novamira_result=' . $action));
        exit();
    }
}

function novamira_render_sandbox_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    $result_message = match ($_GET['novamira_result'] ?? null) {
        'delete' => __('File deleted.', domain: 'novamira'),
        'disable' => __('File disabled.', domain: 'novamira'),
        'enable' => __('File enabled.', domain: 'novamira'),
        'exit_safe_mode' => __(
            'Safe mode deactivated. Sandbox files will load on the next request.',
            domain: 'novamira',
        ),
        default => null,
    };
    $sandbox_dir = novamira_get_sandbox_dir(true);
    $is_crashed = file_exists($sandbox_dir . '.crashed');

    ?>
    <div class="wrap">
        <?php novamira_render_admin_header(); ?>
        <h1><?php esc_html_e('Sandbox Files', domain: 'novamira'); ?></h1>
        <?php if ($result_message !== null) { ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html($result_message); ?></p></div>
        <?php } ?>
        <?php if ($is_crashed) { ?>
            <div class="notice notice-error" style="padding: 12px 16px;">
                <p>
                    <strong><?php esc_html_e('Safe mode is active.', domain: 'novamira'); ?></strong>
                    <?php esc_html_e(
                        'A sandbox file caused a fatal error on a previous request. All sandbox files are suspended until you fix or delete the broken file and exit safe mode.',
                        domain: 'novamira',
                    ); ?>
                </p>
                <p>
                    <?php

                    $exit_url = wp_nonce_url(
                        admin_url('admin.php?page=novamira-sandbox&action=exit_safe_mode&file=.crashed'),
                        action: 'novamira_manage_file_.crashed',
                    );
                    ?>
                    <a href="<?php echo esc_url($exit_url); ?>" class="button button-primary"><?php esc_html_e(
                        'Exit Safe Mode',
                        domain: 'novamira',
                    ); ?></a>
                </p>
            </div>
        <?php } ?>
        <?php novamira_render_sandbox_table(); ?>
    </div>
    <?php
}

function novamira_render_sandbox_table(): void
{ ?>
    <p><?php esc_html_e('Manage the files generated by AI agents in the sandbox directory.', domain: 'novamira'); ?></p>
    <table class="wp-list-table widefat fixed striped">
        <thead>
            <tr>
                <th><?php esc_html_e('Filename', domain: 'novamira'); ?></th>
                <th><?php esc_html_e('Status', domain: 'novamira'); ?></th>
                <th><?php esc_html_e('Last Modified', domain: 'novamira'); ?></th>
                <th><?php esc_html_e('Actions', domain: 'novamira'); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php

            $sandbox_dir = novamira_get_sandbox_dir(true);
            $is_crashed = file_exists($sandbox_dir . '.crashed');
            $scanned_files = is_dir($sandbox_dir) ? scandir($sandbox_dir) : false;
            $files = $scanned_files !== false ? array_diff($scanned_files, ['.', '..', '.loading', '.crashed']) : [];

            if ($files === []) {
                echo '<tr><td colspan="4">' . esc_html__('No sandbox files found.', domain: 'novamira') . '</td></tr>';
            }

            $format = novamira_get_datetime_format();

            foreach ($files as $file) {
                if (is_dir($sandbox_dir . $file)) {
                    continue;
                }
                $path = $sandbox_dir . $file;
                $is_disabled = str_ends_with($file, '.disabled');

                $status = match (true) {
                    $is_disabled => __('Disabled', domain: 'novamira'),
                    $is_crashed => __('Suspended', domain: 'novamira'),
                    default => __('Enabled', domain: 'novamira'),
                };

                $mtime = filemtime($path);
                $wp_date = $mtime !== false ? wp_date($format, $mtime) : false;
                $modified = $wp_date !== false ? $wp_date : __('Unknown', domain: 'novamira');

                $base_url = admin_url('admin.php?page=novamira-sandbox');

                $delete_url = wp_nonce_url(
                    $base_url . '&action=delete&file=' . urlencode($file),
                    'novamira_manage_file_' . $file,
                );
                $toggle_action = $is_disabled ? 'enable' : 'disable';
                $toggle_url = wp_nonce_url(
                    $base_url . '&action=' . $toggle_action . '&file=' . urlencode($file),
                    'novamira_manage_file_' . $file,
                );

                ?>
                <tr>
                    <td><strong><?php echo esc_html($file); ?></strong></td>
                    <td><?php echo esc_html($status); ?></td>
                    <td><?php echo esc_html($modified); ?></td>
                    <td>
                        <a href="<?php echo esc_url($toggle_url); ?>" class="button button-small"><?php echo
                            $is_disabled
                                ? esc_html__('Enable', domain: 'novamira')
                                : esc_html__('Disable', domain: 'novamira')
                        ; ?></a>
                        <a href="<?php echo esc_url($delete_url); ?>" class="button button-small" onclick="return confirm('<?php echo
                            esc_js(__('Are you sure you want to delete this file?', domain: 'novamira'))
                        ; ?>');" style="color: #d63638; border-color: #d63638;"><?php esc_html_e(
                            'Delete',
                            domain: 'novamira',
                        ); ?></a>
                    </td>
                </tr>
                <?php
            }
            ?>
        </tbody>
    </table>
    <?php }

function novamira_render_settings_page()
{
    if (!current_user_can('manage_options')) {
        return;
    }

    if (($_POST['novamira_submit'] ?? null) !== null) {
        check_admin_referer('novamira_settings');
        $enabled = ($_POST['novamira_ai_abilities_enabled'] ?? null) !== null;
        update_option('novamira_ai_abilities_enabled', $enabled);
        echo
            '<div class="notice notice-success is-dismissible"><p>'
                . esc_html__('Settings saved.', domain: 'novamira')
                . '</p></div>'
        ;
    }

    $enabled = novamira_is_enabled();

    $abilities = [
        [
            'execute-php',
            __('Code Execution', domain: 'novamira'),
            __('Run PHP code with full access to the WordPress environment.', domain: 'novamira'),
        ],
        [
            'read-file',
            __('Filesystem', domain: 'novamira'),
            __('Read file contents. Binary files are returned as base64.', domain: 'novamira'),
        ],
        [
            'write-file',
            __('Filesystem', domain: 'novamira'),
            __('Create or overwrite files. PHP files are confined to the sandbox directory.', domain: 'novamira'),
        ],
        [
            'edit-file',
            __('Filesystem', domain: 'novamira'),
            __('Apply a targeted text replacement to an existing file.', domain: 'novamira'),
        ],
        [
            'delete-file',
            __('Filesystem', domain: 'novamira'),
            __('Delete files or directories. Critical WordPress directories are protected.', domain: 'novamira'),
        ],
        [
            'disable-file',
            __('Filesystem', domain: 'novamira'),
            __('Disable a sandbox file by renaming it with a .disabled suffix.', domain: 'novamira'),
        ],
        [
            'enable-file',
            __('Filesystem', domain: 'novamira'),
            __('Re-enable a previously disabled sandbox file.', domain: 'novamira'),
        ],
        [
            'list-directory',
            __('Filesystem', domain: 'novamira'),
            __('List directory contents with glob filtering and recursive traversal.', domain: 'novamira'),
        ],
    ];
    ?>
    <div class="wrap">
        <?php novamira_render_admin_header(); ?>
        <h1><?php esc_html_e('Novamira Settings', domain: 'novamira'); ?></h1>
        <form method="post" action="" id="novamira-settings-form">
            <?php wp_nonce_field('novamira_settings'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><?php esc_html_e('AI Abilities', domain: 'novamira'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="novamira_ai_abilities_enabled" value="1" id="novamira-enable-checkbox" <?php checked(
                                checked: $enabled,
                                current: true,
                            ); ?> />
                            <?php esc_html_e('Enable AI Abilities', domain: 'novamira'); ?>
                        </label>
                        <p class="description"><strong style="color:#d63638;"><?php esc_html_e(
                            'Security note:',
                            domain: 'novamira',
                        ); ?></strong> <?php esc_html_e(
                            'When enabled, AI agents can execute PHP code and perform filesystem operations on this site. For development and staging environments only — always keep backups.',
                            domain: 'novamira',
                        ); ?></p>
                        <p class="description"><?php esc_html_e(
                            'For best results, use with capable, instruction-following AI models. Configure your MCP client to require approval before executing tools, and review each tool call before allowing it to run.',
                            domain: 'novamira',
                        ); ?></p>
                        <p class="description"><?php esc_html_e(
                            'You choose the AI model, you provide the API key, you review the output. We provide the plugin.',
                            domain: 'novamira',
                        ); ?></p>
                    </td>
                </tr>
            </table>
            <?php submit_button(
                text: __('Save Settings', domain: 'novamira'),
                type: 'primary',
                name: 'novamira_submit',
            ); ?>
        </form>
        <script>
        document.getElementById('novamira-settings-form').addEventListener('submit', function (e) {
            var cb = document.getElementById('novamira-enable-checkbox');
            if (cb.checked && !cb.defaultChecked) {
                if (!confirm('<?php echo
                    esc_js(__(
                        'AI agents will be able to execute PHP code and access the filesystem. For development and staging environments only. Continue?',
                        domain: 'novamira',
                    ))
                ; ?>')) {
                    e.preventDefault();
                }
            }
        });
        </script>

        <h2><?php esc_html_e('Available Abilities', domain: 'novamira'); ?></h2>
        <p><?php esc_html_e(
            'These MCP tools are exposed to AI agents when AI Abilities are enabled.',
            domain: 'novamira',
        ); ?></p>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th style="width:200px;"><?php esc_html_e('Ability', domain: 'novamira'); ?></th>
                    <th style="width:130px;"><?php esc_html_e('Category', domain: 'novamira'); ?></th>
                    <th><?php esc_html_e('Description', domain: 'novamira'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($abilities as [$slug, $category, $description]): ?>
                    <tr>
                        <td><code><?php echo esc_html($slug); ?></code></td>
                        <td><?php echo esc_html($category); ?></td>
                        <td><?php echo esc_html($description); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <?php
}
